/*
   Copyright (C) 1997-2004 John Coppens (john@jcoppens.com)
  
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <unistd.h>
#include "i2c.h"

// Initialize the hardware port, and set the i2c address of this device.         
//  Input paddr: parallel port address                                            
//        iaddr: i2c device address (lower 7 bits)                                

i2c_port::i2c_port(unsigned paddr, unsigned iaddr): 
  par_port (paddr)
{
  i2c_addr = iaddr;
}

i2c_port::~i2c_port(void)
{
}

void
i2c_port::SetSDA(bool state)
{
  par_port::SetDataBit(I2C_SDAOUT, state ^ I2C_INV_SDAOUT);
}

void
i2c_port::SetSCL(bool state) 
{
  par_port::SetControlBit(I2C_SCLOUT, state ^ I2C_INV_SCLOUT);
}

bool
i2c_port::GetSDA(void)
{
  return par_port::GetStatusBit(I2C_SDAIN) ^ I2C_INV_SDAIN;
}

bool
i2c_port::GetSCL(void)
{
  return par_port::GetStatusBit(I2C_SCLIN) ^ I2C_INV_SCLIN;
}


void
i2c_port::RaiseSCL(void)
{
  SetSCL(true);
  while (!GetSCL()) {};
}

void
i2c_port::Start(void)
{
  SetSCL(false);
  SetSDA(true);
  SetSCL(true);
  usleep(I2C_DEL_90);
  SetSDA(false);
  usleep(I2C_DEL_91);
  SetSCL(false);
}

void
i2c_port::Stop(void)
{
  SetSCL(false);
  SetSDA(false);
  SetSCL(true);
  usleep(I2C_DEL_92);   
  SetSDA(true);
  usleep(I2C_DEL_93);
}


bool
i2c_port::ByteSent(unsigned b)
{
  int bit;
  bool res;

  for (bit = 7; bit >= 0; bit--) {
    SetSDA( (b & (1 << bit)) != 0);
    RaiseSCL();
    SetSCL(false);
  }
  SetSDA(true);
  RaiseSCL();
  res = not GetSDA();
  SetSCL(false);

  return res;
}


bool
i2c_port::ByteReceived(unsigned *b, bool do_ack)
{
  int bit;
  unsigned data;

  SetSDA(true);
  for (bit = 7; bit >=0; bit--) {
    RaiseSCL();
    data <<= 1;
    if (GetSDA())
      data |= 1;
    SetSCL(false);
  }
  SetSDA(not do_ack);
  RaiseSCL();
  SetSCL(false);
  SetSDA(true);

  *b = data;
  return true;
}
