#!/usr/bin/env python3
# -*- Mode: Python; coding: utf-8; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- 
#
# main.py
# Copyright (C) 2015 Unknown <root@hp425>
# 
# sync_project is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# sync_project is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

from gi.repository import Gtk, GdkPixbuf, Gdk
import os, sys, glob, os.path as osp, json
import pdb


#Comment the first line and uncomment the second before installing
#or making the tarball (alternatively, use project variables)
UI_FILE = "sync_project.ui"
#UI_FILE = "/usr/local/share/sync_project/ui/sync_project.ui"

def fill_file_treeview(path, trview):
    """ Fill the file list with the names of the files in the selected
        folder.
    """
    store = trview.get_model()
    for f in glob.glob(path + '/*'):
        store.append((osp.basename(f), "", ""))

def prepare_file_treeview(trview):
    """ Prepare the treeviews for the file lists. a ListStore is
        created for each case, with three columns in the TreeView.
    """
    store = Gtk.ListStore(str, str, str)
    trview.set_model(store)
    sel = trview.get_selection()
    sel.set_mode(Gtk.SelectionMode.MULTIPLE)
    
    renderer = Gtk.CellRendererText()
    col = Gtk.TreeViewColumn("File name", renderer, text = 0)
    trview.append_column(col)
    col = Gtk.TreeViewColumn("File size", renderer, text = 1)
    trview.append_column(col)
    col = Gtk.TreeViewColumn("Modified",  renderer, text = 2)
    trview.append_column(col)

def prepare_project_cbbox(cbbox):
    """ Initial preparation of the project selection combobox.
        This creates a liststore, width two columns. The first column
        will contain the project's ID, and the second column will con-
        tain a JSON-formatted record, with the path and file list, both
        for local and remote configs
    """
    store = Gtk.ListStore(str, str)
    cbbox.set_model(store)
    
    renderer = Gtk.CellRendererText()
    cbbox.pack_start(renderer, True)
    cbbox.add_attribute(renderer, "text", 0)
    store.append(("John", ""))
    store.append(("Maria", ""))
    cbbox.set_active(0)
    
def add_project(objs):
    """ Add the current configuration (paths and selected files, both
        local and remote) to the project_cbbox's storage.
    """
    
    ###### Falta agregar el dialogo para solicitar el nombre del
    #      proyecto, controlar si no esta vacio, o repetido
    proj_name = "Test project"

    local_path  = objs["local_filechooserbutton"].get_filename()
    remote_path = objs["remote_filechooserbutton"].get_filename()
    
    # Get the selected names in the local file list
    sel = objs["local_treeview"].get_selection()
    local_files = []
    sel.selected_foreach(lambda m, p, i: local_files.append(m.get(i, 0)[0]))
    
    # Get the selected names in the remote file list
    sel = objs["remote_treeview"].get_selection()
    remote_files = []
    sel.selected_foreach(lambda m, p, i: remote_files.append(m.get(i, 0)[0]))
    
    js_rec = json.dumps([{"project_id"  : proj_name,
                          "local_path"  : local_path,
                          "remote_path" : remote_path,
                          "local_files" : local_files,
                          "remote_files": remote_files}])
    
    # Add to the project combo box
    store = objs["project_cbbox"].get_model()
    store.append((proj_name, js_rec))
    
   
class GUI:
    """ GUI loads the UI file, connects the defined signals with 
        their handlers, and creates a dict() with object references
        to the widgets we need in the rest of the program. The dict()
        indices are the same as the widget names.
    """
    def __init__(self):

        self.builder = Gtk.Builder()
        self.builder.add_from_file(UI_FILE)
        self.builder.connect_signals(self)

        self.objects = {}
        for ref in ("MainWindow",
                    "project_cbbox",
                    "local_filechooserbutton",
                    "local_treeview",
                    "remote_filechooserbutton",
                    "remote_treeview"):
            self.objects[ref] = self.builder.get_object(ref)

        self.objects["MainWindow"].show_all()

    def on_MainWindow_delete_event(self, window, event):
        Gtk.main_quit()
        
    def on_file_quit_item_activate(self, item):
        Gtk.main_quit()
        
    # Handlers for events from local filechooser and treeview
    def on_local_filechooserbutton_file_set(self, fc):
        fill_file_treeview(fc.get_filename(),
                           self.objects["local_treeview"])
        
    def on_local_treeview_realize(self, trview):
        prepare_file_treeview(trview)

    # Handlers for events from remote filechooser and treeview
    def on_remote_filechooserbutton_file_set(self, fc):
        fill_file_treeview(fc.get_filename(),
                           self.objects["remote_treeview"])

    def on_remote_treeview_realize(self, trview):
        prepare_file_treeview(trview)
        
    # Handlers for events from project management
    def on_project_cbbox_changed(self, cbbox):
        pass
        
    def on_project_cbbox_realize(self, cbbox):
        prepare_project_cbbox(cbbox)
        
    def on_add_project_btn_clicked(self, btn):
        add_project(self.objects)

def main():
    app = GUI()
    Gtk.main()

if __name__ == "__main__":
	sys.exit(main())

